<?php
/**
 * Workflow Logs API class
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Rest;

use COCO\PolylangAutotranslate\Actions_Logic;
use COCO\PolylangAutotranslate\Chunks\Chunk_Manager;
use COCO\PolylangAutotranslate\Workflow_Logs\Workflow_Log_Crud;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Workflow Logs API class
 */
class Workflow_Logs_Api {

	/**
	 * The namespace for the REST API.
		 */
	const API_NAMESPACE = 'coco-polylang-autotranslate/v1';

	/**
	 * Initialize the REST API endpoints.
	 */
	public static function init(): void {
		add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
	}

	/**
	 * Register the REST API routes.
	 */
	public static function register_routes(): void {
		register_rest_route(
			self::API_NAMESPACE,
			'/workflow-log/(?P<id>\d+)',
			[
				'methods'             => 'POST',
				'callback'            => [ __CLASS__, 'update_workflow_log' ],
				'permission_callback' => '__return_true', // TODO: Add a proper permission check
			]
		);
	}

	/**
	 * Update a workflow log.
	 *
	 * @param \WP_REST_Request $request The REST request.
	 * @return \WP_REST_Response
	 */
	public static function update_workflow_log( \WP_REST_Request $request ): \WP_REST_Response {
		$workflow_log_id = $request->get_param( 'id' );
		$status          = $request->get_param( 'status' );
		$message         = $request->get_param( 'message' );

		if ( empty( $status ) ) {
			return new \WP_REST_Response( [ 'error' => 'Status is required.' ], 400 );
		}

		$updated = Workflow_Log_Crud::update_workflow_log_status( $workflow_log_id, $status, $message, $request->get_params() );

		if ( $updated ) {
			$extra_data            = null;
			$n8n_last_step         = $request->get_param( 'n8n-results' );
			$is_definitive_success = 1;
			$return                = [
				'success'          => $is_definitive_success,
				'original_request' => $request->get_params(),
			];

			// If we arrived to the last n8n node, it will return the post ids where we should have the chunks from.
			if ( ! empty( $n8n_last_step ) && is_array( $n8n_last_step ) ) { // set up the post ids to assemble the chunks

				$extra_data = [];
				foreach ( $n8n_last_step as $chunk_result ) {
					$is_success                                    = $chunk_result['success'];
					$is_definitive_success                         = $is_definitive_success && $is_success;
					$extra_data[ $chunk_result['target_post_id'] ] = $extra_data[ $chunk_result['target_post_id'] ] ?? 1 && $is_success;
				}
				// now extra data has unique post ids a keys.
				// We proceed to assemble and save in the post_content of the target posts
				foreach ( $extra_data as $post_id => $is_success ) {
					$is_success             = Chunk_Manager::save_post_content_from_chunks( $post_id ) && $is_success;
					$is_definitive_success  = $is_definitive_success && $is_success;
					$extra_data[ $post_id ] = $is_success;
				}


				$last_log_update = Workflow_Log_Crud::update_workflow_log_status(
					$workflow_log_id,
					$is_definitive_success ? 'wp-completed' : 'wp-completed-with-fail',
					'WP has received the last n8n node notification and has tried to assemble the chunks ',
					$request->get_params()
				);

				$return = [
					'success'                  => $is_definitive_success,
					'last-workflow-log-update' => $last_log_update,
					'original_request'         => $request->get_params(),
				];
			}
			$return = is_array( $extra_data ) ? array_merge( $return, [ 'extra_data' => $extra_data ] ) : $return;
			return new \WP_REST_Response( $return, 200 );
		} else {
			return new \WP_REST_Response( [ 'error' => "Workflow log with id $workflow_log_id not found." ], 404 );
		}
	}
}
