<?php
/**
 * Chunk Transients API class
 *
 * Handles temporary storage of content chunks during translation workflow.
 *
 * @package CocoPolylangAutotranslate
 *
 * @usage
 * Endpoints:
 * - POST /wp-json/coco-polylang-autotranslate/v1/save-chunk
 *   Saves a content chunk to temporary storage
 *   Parameters:
 *   - workflow_id: int (required)
 *   - post_id: int (required)
 *   - chunk_index: int (required)
 *   - chunk_content: string (required)
 *
 * - GET /wp-json/coco-polylang-autotranslate/v1/get-chunk
 *   Retrieves a previously saved content chunk
 *   Parameters:
 *   - post_id: int (required)
 *   - chunk_index: int (required)
 */

namespace COCO\PolylangAutotranslate\Rest;

use COCO\PolylangAutotranslate\Chunks\Chunk_Manager;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Chunk Transients API class
 */
class Chunk_Transients_Api {

	/**
	 * The namespace for the REST API.
	 */
	const API_NAMESPACE = 'coco-polylang-autotranslate/v1';

	/**
	 * Initialize the REST API endpoints.
	 */
	public static function init(): void {
		add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
	}

	/**
	 * Register the REST API routes.
	 */
	public static function register_routes(): void {
		register_rest_route(
			self::API_NAMESPACE,
			'/save-chunk',
			[
				'methods'             => 'POST',
				'callback'            => [ __CLASS__, 'save_chunk' ],
				'permission_callback' => function () {
					return current_user_can( 'manage_options' );
				},
			]
		);

		register_rest_route(
			self::API_NAMESPACE,
			'/get-chunk',
			[
				'methods'             => 'GET',
				'callback'            => [ __CLASS__, 'get_chunk' ],
				'permission_callback' => function () {
					return current_user_can( 'manage_options' );
				},
			]
		);

		register_rest_route(
			self::API_NAMESPACE,
			'/assemble-and-save-post-content',
			[
				'methods'             => 'POST',
				'callback'            => [ __CLASS__, 'assemble_and_save_post_content' ],
				'permission_callback' => function () {
					return current_user_can( 'manage_options' );
				},
			]
		);
	}

	/**
	 * Save a content chunk to a transient.
	 *
	 * @param \WP_REST_Request $request The REST request.
	 * @return \WP_REST_Response
	 */
	public static function save_chunk( \WP_REST_Request $request ): \WP_REST_Response {
		$workflow_id   = $request->get_param( 'workflow_id' );
		$post_id       = $request->get_param( 'post_id' );
		$chunk_index   = $request->get_param( 'chunk_index' );
		$chunk_content = $request->get_param( 'chunk_content' );

		if ( empty( $workflow_id ) || empty( $post_id ) || ! isset( $chunk_index ) || empty( $chunk_content ) ) {
			return new \WP_REST_Response( [ 'error' => 'Missing required parameters.' ], 400 );
		}

		$transient_key = Chunk_Manager::get_chunk_transient_key( $post_id, $chunk_index );

		set_transient( $transient_key, $chunk_content, 12 * HOUR_IN_SECONDS );

		return new \WP_REST_Response( [ 'success' => true ], 200 );
	}

	/**
	 * Get a content chunk from a transient.
	 *
	 * @param \WP_REST_Request $request The REST request.
	 * @return \WP_REST_Response
	 */
	public static function get_chunk( \WP_REST_Request $request ): \WP_REST_Response {
		$post_id     = $request->get_param( 'post_id' );
		$chunk_index = $request->get_param( 'chunk_index' );

		if ( empty( $post_id ) || ! isset( $chunk_index ) ) {
			return new \WP_REST_Response( [ 'error' => 'Missing required parameters.' ], 400 );
		}

		$transient_key = Chunk_Manager::get_chunk_transient_key( $post_id, $chunk_index );
		$chunk_content = get_transient( $transient_key );

		if ( false === $chunk_content ) {
			return new \WP_REST_Response( [ 'error' => 'Chunk not found.' ], 404 );
		}

		return new \WP_REST_Response( [ 'chunk_content' => $chunk_content ], 200 );
	}

	/**
	 * Assemble and save the post content from chunks.
	 *
	 * @param \WP_REST_Request $request The REST request.
	 * @return \WP_REST_Response
	 */
	public static function assemble_and_save_post_content( \WP_REST_Request $request ): \WP_REST_Response {
		$post_id = $request->get_param( 'post_id' );

		if ( empty( $post_id ) ) {
			return new \WP_REST_Response( [ 'error' => 'Missing required parameters.' ], 400 );
		}

		Chunk_Manager::save_post_content_from_chunks( $post_id );

		return new \WP_REST_Response( [ 'success' => true ], 200 );
	}
}
