<?php
/**
 * Class to extend REST API endpoints with Polylang information.
 * We will add the following fields to the endpoint:
 *
 * <baseurl>/wp-json/wp/v2/pages/232
 *
 * "polylang_language": "en",
 * "polylang_translations": {
 *         "es": 1074,
 *         "fr": 1557
 *     },
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Polylang;

/**
 * Class REST_API_Extension
 * Extends the WordPress REST API to include Polylang language information for posts and pages.
 */
class REST_API_Extension {

	/**
	 * Constructor. Initialize hooks.
	 */
	public static function init(): void {
		// Check if Polylang is active before registering hooks
		add_action( 'plugins_loaded', [ __CLASS__, 'setup_hooks' ] );
	}

	/**
	 * Setup hooks if Polylang is active.
	 */
	public static function setup_hooks(): void {
		// Check if Polylang is active
		if ( ! function_exists( 'pll_current_language' ) ) {
			return;
		}
		// Register REST field for posts
		add_action( 'rest_api_init', [ __CLASS__, 'register_polylang_rest_fields' ] );
	}

	/**
	 * Register REST API fields for Polylang information.
	 */
	public static function register_polylang_rest_fields(): void {
		// Post types to extend
		$post_types = [ 'post', 'page' ];

		foreach ( $post_types as $post_type ) {
			// Add polylang_language field
			register_rest_field(
				$post_type,
				'polylang_language',
				[
					'get_callback'    => [ __CLASS__, 'get_post_language' ],
					'update_callback' => null,
					'schema'          => [
						'description' => __( 'Language slug for the post', 'coco-polylang-autotranslate' ),
						'type'        => 'string',
					],
				]
			);

			// Add polylang_translations field
			register_rest_field(
				$post_type,
				'polylang_translations',
				[
					'get_callback'    => [ __CLASS__, 'get_post_translations' ],
					'update_callback' => null,
					'schema'          => [
						'description' => __( 'Translations of the post in other languages', 'coco-polylang-autotranslate' ),
						'type'        => 'object',
						'properties'  => [
							'language_slug' => [
								'type' => 'integer',
							],
						],
					],
				]
			);
		}
	}

	/**
	 * Get the language slug for a post.
	 *
	 * @param array $post Post object as array.
	 * @return string|null Language slug or null if not found.
	 */
	public static function get_post_language( array $post ): ?string {
		$post_id = $post['id'];
		return function_exists( 'pll_get_post_language' ) ? pll_get_post_language( $post_id, 'slug' ) : null;
	}

	/**
	 * Get translations of a post.
	 *
	 * @param array $post Post object as array.
	 * @return array Associative array with language slugs as keys and post IDs as values.
	 */
	public static function get_post_translations( array $post ): array {
		$post_id      = $post['id'];
		$translations = [];

		if ( function_exists( 'pll_get_post_translations' ) ) {
			$translation_ids = pll_get_post_translations( $post_id );

			foreach ( $translation_ids as $lang => $translation_id ) {
				// Skip the current post
				if ( $translation_id !== $post_id ) {
					$translations[ $lang ] = $translation_id;
				}
			}
		}

		return $translations;
	}
}
