<?php
/**
 * N8N Connection class
 *
 * Handles communication with the n8n webhook
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\N8N;

use COCO\PolylangAutotranslate\Settings_Page;
use COCO\PolylangAutotranslate\Polylang\Polylang_Helpers;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * N8N Connection class
 */
class N8N_Connection {

	/**
	 * Sends data to the n8n webhook.
	 *
	 * @param int   $workflow_log_id The ID of the workflow log.
	 * @param array $params The parameters to send to n8n.
	 * @return bool|\WP_Error True on success, WP_Error on failure.
	 */
	public static function send_to_n8n( int $workflow_log_id, array $params ): bool|\WP_Error {
		$n8n_webhook_url = get_option( Settings_Page::OPTION_N8N_WEBHOOK, '' );

		if ( ! empty( $params['run-test'] ) ) {
			$n8n_webhook_url = str_replace( '/webhook/', '/webhook-test/', $n8n_webhook_url );
		}

		if ( empty( $n8n_webhook_url ) ) {
			return new \WP_Error( 'n8n_webhook_url_not_set', __( 'n8n webhook URL is not set in plugin settings.', 'coco-polylang-autotranslate' ) );
		}

		$payload = array_merge( [ 'workflow_id' => $workflow_log_id ], $params );

		$response = wp_remote_post(
			$n8n_webhook_url,
			[
				'body'    => wp_json_encode( $payload ),
				'headers' => [
					'Content-Type' => 'application/json',
				],
			]
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		if ( 200 !== $response_code ) {
			$response_body = wp_remote_retrieve_body( $response );
			return new \WP_Error( 'n8n_request_failed', $response_body );
		}

		return true;
	}
}
