<?php
/**
 * Chunk Manager class
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Chunks;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Chunk Manager class
 */
class Chunk_Manager {

	/**
	 * Get the transient key for a chunk.
	 *
	 * @param int $post_id The post ID.
	 * @param int $chunk_index The chunk index.
	 * @return string The transient key.
	 */
	public static function get_chunk_transient_key( int $post_id, int $chunk_index ): string {
		return sprintf( 'coco_pa_chunk_%d_%d', $post_id, $chunk_index );
	}

	/**
	 * Assemble the post content from chunks.
	 *
	 * @param int $post_id The post ID.
	 * @return string|false The post content or false if no chunks found.
	 */
	public static function assemble_post_content_from_chunks( int $post_id ): string|false {
		$post_content = '';
		$chunk_index  = 0;
		$found_chunks = false;

		while ( true ) {
			$transient_key = self::get_chunk_transient_key( $post_id, $chunk_index );
			$chunk_content = get_transient( $transient_key );

			if ( false === $chunk_content ) {
				break;
			}

			$post_content .= $chunk_content;
			$chunk_index++;
			$found_chunks = true;
		}

		return $found_chunks ? $post_content : false;
	}

	/**
	 * Save the post content from chunks.
	 *
	 * @param int $post_id The post ID.
	 * @return bool True if content was successfully assembled and saved, false otherwise.
	 */
	public static function save_post_content_from_chunks( int $post_id ): bool {
		$post_content = self::assemble_post_content_from_chunks( $post_id );

		if ( empty( $post_content ) ) {
			return false;
		}

		$update_result = wp_update_post(
			[
				'ID'           => $post_id,
				'post_content' => $post_content,
			]
		);

		return ! is_wp_error( $update_result ) && $update_result > 0;
	}
}
