<?php
/**
 * Workflow Logs Table class
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Admin;

use COCO\PolylangAutotranslate\Chunks\Chunk_Manager;

use COCO\PolylangAutotranslate\Rest\Workflow_Logs_Api;
use COCO\PolylangAutotranslate\Workflow_Logs\Workflow_Log_Crud;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Workflow Logs Table class
 */
class Workflow_Logs_Table extends \WP_List_Table {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct(
			[
				'singular' => __( 'Workflow Log', 'coco-polylang-autotranslate' ),
				'plural'   => __( 'Workflow Logs', 'coco-polylang-autotranslate' ),
				'ajax'     => false,
			]
		);
	}

	/**
	 * Get the columns.
	 *
	 * @return array
	 */
	public function get_columns(): array {
		return [
			'id'               => __( 'ID', 'coco-polylang-autotranslate' ),
			'status'           => __( 'Status', 'coco-polylang-autotranslate' ),
			'message'          => __( 'Message', 'coco-polylang-autotranslate' ),
			'post_ids'         => __( 'Post IDs', 'coco-polylang-autotranslate' ),
			'target_languages' => __( 'Target Languages', 'coco-polylang-autotranslate' ),
			'updated_at'       => __( 'Last Updated', 'coco-polylang-autotranslate' ),
		];
	}

	/**
	 * Get the sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns(): array {
		return [
			'id'         => [ 'id', true ],
			'status'     => [ 'status', false ],
			'updated_at' => [ 'updated_at', false ],
		];
	}

	/**
	 * Prepare the items.
	 */
	public function prepare_items(): void {
		$this->_column_headers = [ $this->get_columns(), [], $this->get_sortable_columns() ];

		$data = Workflow_Log_Crud::get_workflow_logs();

		// Sorting
		$orderby = $_GET['orderby'] ?? 'id';
		$order   = $_GET['order'] ?? 'desc';

		usort(
			$data,
			function ( $a, $b ) use ( $orderby, $order ) {
				$result = 0;
				if ( $orderby === 'id' || $orderby === 'updated_at' ) {
					$result = $a[ $orderby ] <=> $b[ $orderby ];
				} else {
					$result = strcmp( $a[ $orderby ], $b[ $orderby ] );
				}
				return $order === 'asc' ? $result : -$result;
			}
		);

		// Pagination
		$per_page     = 20;
		$current_page = $this->get_pagenum();
		$total_items  = count( $data );

		$this->set_pagination_args(
			[
				'total_items' => $total_items,
				'per_page'    => $per_page,
			]
		);

		$this->items = array_slice( $data, ( ( $current_page - 1 ) * $per_page ), $per_page );
	}

	/**
	 * Default column.
	 *
	 * @param array  $item The item.
	 * @param string $column_name The column name.
	 * @return mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'id':
				$data = (string) $item['id'];
				if ( isset( $item['n8n-workflow-id'] ) ) {
					$data .= sprintf(
						'<p class="muted">n8n id: <a href="%s" target="_blank">%s</a></p>',
						( $item['params']['debug-url'] ?? '#' ),
						$item['n8n-workflow-id']
					);
				}
				if ( ! empty( $item['params']['run-test'] ) ) {
					$data .= '<br/><strong style="color:orange;">test</strong>';
				}
				return $data;
			case 'status':
				return sprintf( '<div style="max-width: 100px;">%s</div>', $item['status'] );
			case 'message':
				$message  = isset( $item['message'] ) ? esc_html( $item['message'] ) : '';
				$params   = Workflow_Log_Crud::process_sensitive_params( $item['params'] );
				$message .= '<div class="payload">';
				foreach ( $params as $key => $value ) {
						$message .= sprintf(
							'<strong>%s</strong>: <pre>%s</pre>',
							esc_html( $key ),
							print_r( $value, true )
						);
				}
				$message .= '</div>';
				return sprintf(
					'<div class="message-truncated">%s</div>',
					$message
				);
			case 'post_ids':
				if ( empty( $item['params']['post_ids'] ) || ! is_array( $item['params']['post_ids'] ) ) {
					return 'Error. post_ids no an array:  ';
				}
				return implode( ', ', array_map( function ( $post_id ) {
					$post = get_post( $post_id );
					if ( $post ) {
							$edit_link = get_edit_post_link( $post_id );
							return sprintf(
								'<a href="%s">%s</a> (%d) <br/>',
								esc_url( $edit_link ),
								esc_html( $post->post_title ),
								$post_id
							);
					}
						return $post_id;
				}, is_array( $item['params']['post_ids'] ) ? $item['params']['post_ids'] : [] ) );
			case 'target_languages':
				$target_languages = $item['params']['target_languages'] ?? [];
				$post_ids         = $item['params']['post_ids'] ?? [];
				$output           = '';

				foreach ( $post_ids as $post_id ) {
					$translations = pll_get_post_translations( $post_id );

					foreach ( $target_languages as $order => $lang ) {
						$output .= $order ? ', ' : '';
						if ( isset( $translations[ $lang ] ) ) {
							$translated_post_id = $translations[ $lang ];
							$edit_link          = get_edit_post_link( $translated_post_id );

							$output .= sprintf(
								'<a href="%s">%s</a> (%s)',
								esc_url( $edit_link ),
								esc_html( $translated_post_id ),
								esc_html( $lang )
							);
							$output .= self::display_chunk_transients( $translated_post_id );
						} else {
							$output .= esc_html( $lang ) . '(❌ no translation post)';
						}
					}
					$output .= '<br/>';
				}
				return $output;
			case 'updated_at':
				return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $item['updatedAt'] );
			default:
				return print_r( $item, true );
		}
	}

	/**
	 * Display chunk transients for a post.
	 *
	 * @param int $post_id The post ID.
	 * @return string
	 */
	private static function display_chunk_transients( int $post_id ): string {
		$chunk_index = 0;
		$output      = '';
		while ( true ) {
			$transient_key = Chunk_Manager::get_chunk_transient_key( $post_id, $chunk_index );
			$chunk_content = get_transient( $transient_key );

			if ( false === $chunk_content ) {
				break;
			}

			$output .= sprintf( '<li>Chunk %d: %d chars</li>', $chunk_index, strlen( $chunk_content ) );
			$chunk_index++;
		}

		if ( ! empty( $output ) ) {
			$nonce_url = wp_nonce_url(
				admin_url( 'admin-post.php?action=coco_pa_assemble_chunks&post_id=' . $post_id ),
				'coco_pa_assemble_chunks_' . $post_id
			);
			$output   .= sprintf(
				'<li><a href="%s" class="button button-primary">%s</a></li>',
				esc_url( $nonce_url ),
				esc_html__( 'Assemble', 'coco-polylang-autotranslate' )
			);
			return '<ul style="font-size: 0.8em; color: #666;">' . $output . '</ul>';
		}

		return '';
	}
}

// Add custom CSS for message expansion
add_action( 'admin_head', function () {
	?>
		<style>
			.message-truncated {
				max-height: 100px;
				white-space: nowrap;
				overflow: hidden;
				text-overflow: ellipsis;
				cursor: pointer;

				.payload {
					line-height: 1;
					color:rgb(119, 116, 116);

				}
				pre {
					line-height: 1;
					font-size: 0.6rem;
					margin: 0 0 1rem 1rem;
				}
			}
			.message-expanded {
				white-space: normal;
				max-height: none;
			}
			.expand-message {
				color: #0073aa;
				text-decoration: underline;
				cursor: pointer;
				margin-left: 5px;
			}
			.column-message {
				width: 300px !important;
			}
			.column-id {
				max-width: 100px !important;
				width: 100px !important;
			}
			.column-status {
				max-width: 100px !important;
				width: 100px !important;
			}
			.updated_at {
				max-width: 100px !important;
				width: 100px !important;
			}
		</style>
		<?php
} );

// Add JavaScript for message expansion
add_action( 'admin_footer', function () {
	?>
		<script>
		document.addEventListener('DOMContentLoaded', function() {
			document.querySelectorAll('.message-truncated').forEach(function(element) {
				element.addEventListener('click', function() {
					this.classList.toggle('message-expanded');
				});
			});
		});
		</script>
		<?php
} );