<?php
/**
 * Actions Page class
 *
 * Handles the actions page for the Coco Polylang Autotranslate plugin
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate;

use COCO\PolylangAutotranslate\Polylang\Polylang_Helpers;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Actions Page class
 */
class Actions_Page {

	/**
	 * Initialize the actions page
	 *
	 * @return void
	 */
	public static function init(): void {
		// Initialize is handled by Actions_Logic
	}

	/**
	 * Get valid post types for the dropdown
	 *
	 * @return array
	 */
	public static function get_valid_post_types(): array {
		// For now, hardcoded as requested
		return [ 'page', 'post' ];
	}

	/**
	 * Get post type labels for display
	 *
	 * @return array
	 */
	public static function get_post_type_labels(): array {
		return [
			'post' => __( 'Posts', 'coco-polylang-autotranslate' ),
			'page' => __( 'Pages', 'coco-polylang-autotranslate' ),
		];
	}

	/**
	 * Render the actions page
	 *
	 * @return void
	 */
	public static function render(): void {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Get current selected post type
		$current_post_type = Actions_Logic::get_selected_post_type();
		$valid_post_types  = self::get_valid_post_types();
		$post_type_labels  = self::get_post_type_labels();

		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<?php
			// Display admin notices (verifying the GET first)
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$settings_updated = sanitize_text_field( $_GET['settings-updated'] ?? '' );
			if ( ! empty( $settings_updated ) && 'true' === $settings_updated ) {
				settings_errors( 'coco_pa_actions_notices' );
			}
			?>

			<div class="coco-polylang-autotranslate-actions-grid">
				<!-- Post Type Selection Card -->
				<div class="coco-polylang-autotranslate-action-card">
					<h2><?php esc_html_e( 'Post Type Selection', 'coco-polylang-autotranslate' ); ?></h2>
					<p><?php esc_html_e( 'Choose the post type that the plugin will work with for autotranslation actions.', 'coco-polylang-autotranslate' ); ?></p>

					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="coco-pa-post-type-form">
						<?php wp_nonce_field( 'coco_pa_change_post_type', 'coco_pa_nonce' ); ?>
						<input type="hidden" name="action" value="coco_pa_change_post_type">

						<div class="form-group">
							<label for="selected_post_type">
								<?php esc_html_e( 'Select Post Type:', 'coco-polylang-autotranslate' ); ?>
							</label>
							<div class="input-group">
								<select name="selected_post_type" id="selected_post_type" class="regular-text" required>
									<?php foreach ( $valid_post_types as $post_type ) : ?>
										<option value="<?php echo esc_attr( $post_type ); ?>" <?php selected( $current_post_type, $post_type ); ?>>
											<?php echo esc_html( $post_type_labels[ $post_type ] ?? ucfirst( $post_type ) ); ?>
										</option>
									<?php endforeach; ?>
								</select>
								<button type="submit" class="button button-primary">
									<?php esc_html_e( 'Update', 'coco-polylang-autotranslate' ); ?>
								</button>
							</div>
						</div>
					</form>

					<div class="current-selection">
						<p class="description">
							<strong><?php esc_html_e( 'Current selection:', 'coco-polylang-autotranslate' ); ?></strong>
							<?php echo esc_html( $post_type_labels[ $current_post_type ] ?? ucfirst( $current_post_type ) ); ?>
						</p>
					</div>

					<?php self::render_polylang_status(); ?>
				</div>

				<!-- Future Actions Card -->
				<div class="coco-polylang-autotranslate-action-card">
					<h2><?php esc_html_e( 'Autotranslation Actions', 'coco-polylang-autotranslate' ); ?></h2>
					<?php self::render_posts_list(); ?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render Polylang status information
	 *
	 * @return void
	 */
	private static function render_polylang_status(): void {
		if ( ! Polylang_Helpers::is_polylang_active() ) {
			?>
			<div class="notice notice-warning">
				<p><?php esc_html_e( 'Polylang plugin is not active. Please activate Polylang to use the autotranslation features.', 'coco-polylang-autotranslate' ); ?></p>
			</div>
			<?php
			return;
		}

		$current_language = Polylang_Helpers::get_default_language();
		$languages        = pll_languages_list( [ 'fields' => '' ] );
		?>
		<div class="polylang-status">
			<h3><?php esc_html_e( 'Polylang Status', 'coco-polylang-autotranslate' ); ?></h3>
			<p>
				<strong><?php esc_html_e( 'Current Language:', 'coco-polylang-autotranslate' ); ?></strong>
				<?php echo esc_html( Polylang_Helpers::get_language_name( $current_language ) ); ?>
				<br/>
				<strong><?php esc_html_e( 'All Active Languages:', 'coco-polylang-autotranslate' ); ?></strong>
				<?php
					$languages_names = wp_list_pluck( $languages, 'name' );
					echo wp_kses_post( implode( '</span>, <span>', $languages_names ) );
				?>
				<br/>
				<small class="muted"><?php echo wp_kses_post( sprintf( __( '<b>%s</b> is selected as LLM translator' ), get_option( Settings_Page::OPTION_LLM_MODEL, null ) ?? '-nothing-' ) ); ?></small>
			</p>
		</div>
		<?php
	}

	/**
	 * Render the list of posts/pages for the selected post type.
	 *
	 * @return void
	 */
	private static function render_posts_list(): void {
		if ( ! Polylang_Helpers::is_polylang_active() ) {
			return; // Don't render if Polylang is not active.
		}

		$selected_post_type = Actions_Logic::get_selected_post_type();
		$default_language   = Polylang_Helpers::get_default_language();

		if ( ! $default_language ) {
			?>
			<div class="notice notice-warning">
				<p><?php esc_html_e( 'Could not determine the default Polylang language.', 'coco-polylang-autotranslate' ); ?></p>
			</div>
			<?php
			return;
		}

		$args = [
			'post_type'      => $selected_post_type,
			'lang'           => $default_language, // Get posts in the default language.
			'posts_per_page' => -1, // Get all posts.
			'post_status'    => [ 'publish', 'pending', 'draft', 'future', 'private' ],
			'orderby'        => 'title',
			'order'          => 'ASC',
		];

		$posts_query = new \WP_Query( $args );

		if ( ! $posts_query->have_posts() ) {
			?>
			<p><?php esc_html_e( 'No posts found for the selected post type in the default language.', 'coco-polylang-autotranslate' ); ?></p>
			<?php
			return;
		}

		// Get pre-selected posts from URL if any
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$preselected_posts = isset( $_GET['selected_posts'] ) ? array_map( 'intval', explode( ',', sanitize_text_field( $_GET['selected_posts'] ) ) ) : [];
		?>
		<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="coco-pa-posts-form">
			<?php wp_nonce_field( 'coco_pa_process_posts', 'coco_pa_posts_nonce' ); ?>
			<input type="hidden" name="action" value="coco_pa_process_posts">
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<th class="manage-column column-cb check-column"><input type="checkbox" /></th>
						<th><?php esc_html_e( 'Title', 'coco-polylang-autotranslate' ); ?></th>
						<th><?php esc_html_e( 'Translation Status', 'coco-polylang-autotranslate' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php
					while ( $posts_query->have_posts() ) :
						$posts_query->the_post();
						$post_id      = get_the_ID();
						$translations = pll_get_post_translations( $post_id ); // The array looks like [ es => post_id, ... ]

						?>
						<tr>
							<th scope="row" class="check-column">
								<input type="checkbox" name="selected_posts[]" value="<?php echo esc_attr( $post_id ); ?>" <?php checked( in_array( $post_id, $preselected_posts, true ) ); ?> />
							</th>
							<td><a href="<?php echo esc_url( get_edit_post_link() ); ?>"><?php the_title(); ?></a></td>
							<td>
								<?php
								if ( ! empty( $translations ) ) {
									esc_html_e( 'Translated into:', 'coco-polylang-autotranslate' );
									foreach ( $translations as $lang_code => $translated_post_id ) {
										if ( $lang_code !== $default_language ) {
												printf(
													' <a href="%s">%s</a>',
													esc_url( get_edit_post_link( $translated_post_id ) ),
													esc_html( Polylang_Helpers::get_language_name( $lang_code ) )
												);
												echo ( next( $translations ) && key( $translations ) !== $default_language ) ? ', ' : '';
										}
									}
								} else {
									esc_html_e( 'Not translated', 'coco-polylang-autotranslate' );
								}
								?>
							</td>
						</tr>
					<?php endwhile; ?>
				</tbody>
			</table>
			<?php wp_reset_postdata(); ?>

			<div class="language-selection">
				<h3><?php esc_html_e( 'Translate selected posts into:', 'coco-polylang-autotranslate' ); ?></h3>
				<ul>
					<?php
					$all_languages = pll_languages_list( [ 'fields' => '' ] );
					foreach ( $all_languages as $language ) :
						if ( $language->slug !== $default_language ) :
							?>
							<li>
								<label>
									<input type="checkbox" name="target_languages[]" value="<?php echo esc_attr( $language->slug ); ?>"
									<?php
										// Check if target_languages exists in GET and contains this language
										// phpcs:ignore WordPress.Security.NonceVerification.Recommended
										$default_target_langs = isset( $_GET['target_languages'] ) ? sanitize_text_field( $_GET['target_languages'] ) : '';
										$selected_languages   = ! empty( $default_target_langs ) ? array_map( 'sanitize_text_field', explode( ',', $default_target_langs ) ) : [];
										checked( in_array( $language->slug, $selected_languages, true ) );
									?>
									/>
									<?php echo esc_html( $language->name ); ?>
								</label>
							</li>
							<?php
						endif;
					endforeach;
					?>
				</ul>
			</div>

			<div class="test-webhook-option">
				<label>
					<input type="checkbox" name="run_test_webhook" value="yes" />
					<?php esc_html_e( 'Execute test webhook in N8N', 'coco-polylang-autotranslate' ); ?>
				</label>
			</div>

			<p class="submit">
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php esc_attr_e( 'Process Selected', 'coco-polylang-autotranslate' ); ?>">
			</p>
		</form>
		<?php
	}
}
