<?php
/**
 * Workflow Log Crud class
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Workflow_Logs;

use COCO\PolylangAutotranslate\Actions_Logic;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Workflow Log Crud class
 */
class Workflow_Log_Crud {

	/**
	 * Option name to store workflow logs in the database.
	 */
	const WORKFLOW_LOGS_OPTION_NAME = 'coco_pa_workflow_logs';

	/**
	 * Truncates sensitive parameter values. Params sent to n8n which can be returned are:
	 *
	 * @param array $params The parameters to process.
	 * @return array The processed parameters.
	 */
	public static function process_sensitive_params( array $params ): array {
		foreach ( $params as $key => $value ) {
			if ( is_string( $value ) && ( strpos( strtolower( $key ), '_token' ) !== false || strpos( strtolower( $key ), '_pass' ) !== false ) ) {
				$params[ $key ] = substr( $value, 0, 3 ) . '...';
			}
		}
		return $params;
	}

	/**
	 * Adds a new workflow log to the database.
	 *
	 * @param int   $workflow_log_id The ID of the new workflow log.
	 * @param array $params The parameters sent to n8n.
	 * @return void
	 */
	public static function add_workflow_log( int $workflow_log_id, array $params ): void {
		$workflow_logs   = self::get_workflow_logs();
		$timestamp       = time();
		$workflow_logs[] = [
			'id'        => $workflow_log_id,
			'params'    => self::process_sensitive_params( $params ),
			'status'    => 'wp-newborn',
			'message'   => '',
			'updatedAt' => $timestamp,
		];
		// allow to identify a workflow log also by the id given to the workflow in n8n
		if ( isset( $params['n8n-workflow-id'] ) ) {
			$workflow_logs['n8n-workflow-id'] = $params['n8n-workflow-id'];
		}
		update_option( self::WORKFLOW_LOGS_OPTION_NAME, $workflow_logs );
	}

	/**
	 * Retrieves all workflow logs from the database.
	 *
	 * @return array
	 */
	public static function get_workflow_logs(): array {
		return get_option( self::WORKFLOW_LOGS_OPTION_NAME, [] );
	}

	/**
	 * Updates the status of a workflow log.
	 *
	 * @param string|int $workflow_log_id The ID of the workflow log to update.
	 * @param string     $status The new status.
	 * @param string     $message An optional message.
	 * @param array      $params Optional parameters to update.
	 * @return bool True on success, false on failure.
	 */
	public static function update_workflow_log_status( string|int $workflow_log_id, string $status, string $message = '', array $params = [] ): bool {
		$workflow_logs = self::get_workflow_logs();
		foreach ( $workflow_logs as &$workflow_log ) {

			// this is executed only once inthe whole iteration, for the workflow with the give id found.
			if ( (string) $workflow_log['id'] === (string) $workflow_log_id
				|| ( isset( $workflow_log['n8n-workflow-id'] ) && (string) $workflow_log['n8n-workflow-id'] === (string) $workflow_log_id ) ) {
				$workflow_log['status']    = $status;
				$workflow_log['message']   = $message;
				$workflow_log['updatedAt'] = time();
				$workflow_log['params']    = self::process_sensitive_params( array_merge( $workflow_log['params'], $params ) );
				if ( isset( $params['n8n-workflow-id'] ) ) {
					$workflow_log['n8n-workflow-id'] = $params['n8n-workflow-id'];
				}

				update_option( self::WORKFLOW_LOGS_OPTION_NAME, $workflow_logs );

				Workflow_Log_Manager::prune_old_workflow_logs(); // cleanup of db


				// intercepted to complete the task unifying the transients, if everything went ok
				// do_action( 'after_update_workflow_log', $workflow_log_id, $workflow_log );

				return true;
			}
		}
		return false;
	}
}
