<?php
/**
 * Polylang Helpers class
 *
 * Provides helper functions for Polylang integration
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate\Polylang;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Polylang Helpers class
 */
class Polylang_Helpers {

	/**
	 * Check if Polylang is active and available
	 *
	 * @return bool
	 */
	public static function is_polylang_active(): bool {
		return function_exists( 'pll_current_language' ) && function_exists( 'pll_default_language' );
	}

	/**
	 * Get the default/primary language from Polylang
	 *
	 * @return string|null
	 */
	public static function get_default_language(): ?string {
		if ( ! self::is_polylang_active() ) {
			return null;
		}

		return function_exists( 'pll_default_language' ) ? pll_default_language() : null;
	}

	/**
	 * Get the language name from language code
	 *
	 * @param string $language_code Language code (e.g., 'en', 'es').
	 * @return string
	 */
	public static function get_language_name( string $language_code ): string {
		if ( ! self::is_polylang_active() || ! function_exists( 'PLL' ) ) {
			return $language_code;
		}

		$pll = PLL();
		if ( isset( $pll->model->languages ) ) {
			foreach ( $pll->model->languages as $language ) {
				if ( $language->slug === $language_code ) {
					return $language->name;
				}
			}
		}

		return $language_code;
	}

	/**
	 * Get status information about Polylang installation
	 *
	 * @return array Status information array
	 */
	public static function get_polylang_status(): array {
		$status = [
			'is_active'             => self::is_polylang_active(),
			'default_language'      => null,
			'default_language_name' => null,
			'total_languages'       => 0,
		];

		if ( $status['is_active'] ) {
			$default_lang                    = self::get_default_language();
			$status['default_language']      = $default_lang;
			$status['default_language_name'] = $default_lang ? self::get_language_name( $default_lang ) : null;
			
			if ( function_exists( 'pll_languages_list' ) ) {
				$all_languages             = pll_languages_list( [ 'fields' => '' ] );
				$status['total_languages'] = count( $all_languages ?: [] );
			}
		}

		return $status;
	}
}
