<?php
/**
 * Settings Page class
 *
 * Handles the settings page for the Coco Polylang Autotranslate plugin
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings Page class
 */
class Settings_Page {

	/**
	 * Plugin option names as constants
	 */
	const OPTION_N8N_WEBHOOK      = 'coco_pa_n8n_webhook';
	const OPTION_LLM_MODEL        = 'coco_pa_llm_model';
	const OPTION_LLM_API_KEY      = 'coco_pa_llm_api_key';
	const OPTION_WP_APP_PASSWORD  = 'coco_pa_wp_app_password';
	const OPTION_EXTRA_PROMPT     = 'coco_pa_extra_prompt';
	const OPTION_CHUNK_CHAR_LIMIT = 'coco_pa_chunk_char_limit';

	/**
	 * Initialize the settings page
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'admin_init', [ self::class, 'register_settings' ] );
	}

	/**
	 * Register plugin settings
	 *
	 * @return void
	 */
	public static function register_settings(): void {
		register_setting(
			'coco_pa_settings',
			self::OPTION_N8N_WEBHOOK,
			[
				'type'              => 'string',
				'sanitize_callback' => 'esc_url_raw',
			]
		);

		register_setting(
			'coco_pa_settings',
			self::OPTION_LLM_MODEL,
			[
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			]
		);

		register_setting(
			'coco_pa_settings',
			self::OPTION_LLM_API_KEY,
			[
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			]
		);

		register_setting(
			'coco_pa_settings',
			self::OPTION_WP_APP_PASSWORD,
			[
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			]
		);

		register_setting(
			'coco_pa_settings',
			self::OPTION_EXTRA_PROMPT,
			[
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_textarea_field',
			]
		);

		register_setting(
			'coco_pa_settings',
			self::OPTION_CHUNK_CHAR_LIMIT,
			[
				'type'              => 'integer',
				'sanitize_callback' => 'absint',
				'default'           => 500,
			]
		);
	}

	/**
	 * Render the settings page
	 *
	 * @return void
	 */
	public static function render(): void {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Add nonce field for security
		$nonce = wp_create_nonce( 'coco_pa_settings-options' );
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<?php settings_errors( 'coco_pa_messages' ); ?>

			<form action="options.php" method="post">
				<?php
				// Add nonce field and required hidden fields
				wp_nonce_field( 'coco_pa_settings-options' );
				settings_fields( 'coco_pa_settings' );
				?>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_N8N_WEBHOOK ); ?>">
								<?php esc_html_e( 'n8n Webhook URL', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<input
								type="url"
								name="<?php echo esc_attr( self::OPTION_N8N_WEBHOOK ); ?>"
								id="<?php echo esc_attr( self::OPTION_N8N_WEBHOOK ); ?>"
								value="<?php echo esc_url( get_option( self::OPTION_N8N_WEBHOOK ) ); ?>"
								class="regular-text"
							/>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_LLM_MODEL ); ?>">
								<?php esc_html_e( 'LLM Model', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<?php
								$current_option_llm = get_option( self::OPTION_LLM_MODEL ) ?? '';
							?>
							<select
								name="<?php echo esc_attr( self::OPTION_LLM_MODEL ); ?>"
								id="<?php echo esc_attr( self::OPTION_LLM_MODEL ); ?>"
							>
								<option value="" <?php selected( $current_option_llm, '' ); ?>> -- </option>
								<option value="deepseek" <?php selected( $current_option_llm, 'deepseek' ); ?>>
									Deepseek
								</option>
							</select>
							<small class="muted"><?php echo wp_kses_post( sprintf( __( '%s is selected as the translator LLM' ), $current_option_llm ?? '-nothing-' ) ); ?></small>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_LLM_API_KEY ); ?>">
								<?php esc_html_e( 'LLM API Key', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<input
								type="password"
								name="<?php echo esc_attr( self::OPTION_LLM_API_KEY ); ?>"
								id="<?php echo esc_attr( self::OPTION_LLM_API_KEY ); ?>"
								value="<?php echo esc_attr( get_option( self::OPTION_LLM_API_KEY ) ); ?>"
								class="regular-text"
							/>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_WP_APP_PASSWORD ); ?>">
								<?php esc_html_e( 'WordPress Application Password', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<input
								type="password"
								name="<?php echo esc_attr( self::OPTION_WP_APP_PASSWORD ); ?>"
								id="<?php echo esc_attr( self::OPTION_WP_APP_PASSWORD ); ?>"
								value="<?php echo esc_attr( get_option( self::OPTION_WP_APP_PASSWORD ) ); ?>"
								class="regular-text"
							/>
							<br/><small class="muted">User must have login as <b>n8n-workflow</b></small>
							<?php
							$n8n_user = get_user_by( 'login', 'n8n-workflow' );
							if ( $n8n_user ) {
									echo '<br/><small class="success" style="color: green;">✓ User n8n-workflow exists</small>';
							} else {
									echo '<br/><small class="error" style="color: red;">✗ Error: User n8n-workflow does not exist. Please create this user with appropriate permissions.</small>';
							}
							?>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_EXTRA_PROMPT ); ?>">
								<?php esc_html_e( 'Extra Prompt', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<textarea
								name="<?php echo esc_attr( self::OPTION_EXTRA_PROMPT ); ?>"
								id="<?php echo esc_attr( self::OPTION_EXTRA_PROMPT ); ?>"
								class="large-text"
								rows="5"
							><?php echo esc_textarea( get_option( self::OPTION_EXTRA_PROMPT ) ); ?></textarea>
							<p class="description">
								<?php esc_html_e( 'This prompt will be added to the main prompt sent to the LLM.', 'coco-polylang-autotranslate' ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="<?php echo esc_attr( self::OPTION_CHUNK_CHAR_LIMIT ); ?>">
								<?php esc_html_e( 'Chunk Character Limit', 'coco-polylang-autotranslate' ); ?>
							</label>
						</th>
						<td>
							<input
								type="number"
								name="<?php echo esc_attr( self::OPTION_CHUNK_CHAR_LIMIT ); ?>"
								id="<?php echo esc_attr( self::OPTION_CHUNK_CHAR_LIMIT ); ?>"
								value="<?php echo esc_attr( get_option( self::OPTION_CHUNK_CHAR_LIMIT, 500 ) ); ?>"
								class="regular-text"
							/>
							<p class="description">
								<?php esc_html_e( 'The maximum number of characters for each content chunk sent to the LLM.', 'coco-polylang-autotranslate' ); ?>
							</p>
						</td>
					</tr>
				</table>
				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}
}
