<?php
/**
 * Actions Logic class
 *
 * Handles the logic for processing actions in the Coco Polylang Autotranslate plugin
 *
 * @package CocoPolylangAutotranslate
 */

namespace COCO\PolylangAutotranslate;

use COCO\PolylangAutotranslate\N8N\N8N_Connection;
use COCO\PolylangAutotranslate\Settings_Page;
use COCO\PolylangAutotranslate\Workflow_Logs\Workflow_Log_Crud;
use COCO\PolylangAutotranslate\Polylang\Polylang_Helpers;
use COCO\PolylangAutotranslate\Chunks\Chunk_Manager;

// Prevent direct access to this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Actions Logic class
 */
class Actions_Logic {

	/**
	 * Database option name for storing the selected post type
	 */
	const SELECTED_POST_TYPE_OPTION = 'coco_pa_selected_post_type';

	/**
	 * Initialize the actions logic
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'admin_post_coco_pa_change_post_type', [ __CLASS__, 'handle_change_post_type' ] );
		add_action( 'admin_post_coco_pa_process_posts', [ __CLASS__, 'handle_process_posts' ] );
		add_action( 'after_update_workflow_log', [ __CLASS__, 'handle_after_update_workflow_log' ], 10, 2 );
		add_action( 'admin_post_coco_pa_assemble_chunks', [ __CLASS__, 'handle_assemble_chunks' ] );
	}

	/**
	 * Handle the assemble chunks action.
	 *
	 * @return void
	 */
	public static function handle_assemble_chunks(): void {
		$post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;

		if ( ! $post_id || ! check_admin_referer( 'coco_pa_assemble_chunks_' . $post_id ) ) {
			wp_die( esc_html__( 'Invalid request', 'coco-polylang-autotranslate' ) );
		}

		Chunk_Manager::save_post_content_from_chunks( $post_id );

		self::add_admin_notice( 'success', __( 'Post content has been assembled from chunks.', 'coco-polylang-autotranslate' ) );

		wp_safe_redirect( admin_url( 'admin.php?page=coco-polylang-autotranslate-workflow-logs' ) );
		exit;
	}

	/**
	 * Handle the processing of selected posts for translation.
	 *
	 * @return void
	 */
	public static function handle_process_posts(): void {

		// Verify nonce and user capabilities
		$nonce = sanitize_text_field( $_POST['coco_pa_posts_nonce'] ?? '' );
		if ( ! wp_verify_nonce( $nonce ?? '', 'coco_pa_process_posts' )
				|| ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Access denied.', 'coco-polylang-autotranslate' ) );
		}

		// retrieve and sanitize
		$selected_posts   = array_map( 'absint', $_POST['selected_posts'] ?? [] );
		$target_languages = array_map( 'sanitize_text_field', $_POST['target_languages'] ?? [] );

		$extra_args_if_redirect_back = [
			'selected_posts'   => implode( ',', $selected_posts ),
			'target_languages' => implode( ',', $target_languages ),
		];
		if ( empty( $selected_posts ) ) {
			self::add_admin_notice( 'error', __( 'No posts selected for translation.', 'coco-polylang-autotranslate' ) );
			self::redirect_back( $extra_args_if_redirect_back );
			return;
		}

		if ( empty( $target_languages ) ) {
			self::add_admin_notice( 'error', __( 'No target languages selected for translation.', 'coco-polylang-autotranslate' ) );
			self::redirect_back( $extra_args_if_redirect_back );
			return;
		}

		$llm_type      = get_option( Settings_Page::OPTION_LLM_MODEL, '' );
		$llm_api_token = get_option( Settings_Page::OPTION_LLM_API_KEY, '' );
		if ( empty( $llm_type ) || empty( $llm_api_token ) ) {
			self::add_admin_notice( 'error', __( 'LLM Type or API Token not configured in plugin settings.', 'coco-polylang-autotranslate' ) );
			self::redirect_back( $extra_args_if_redirect_back );
			return;
		}

		$first_post          = get_post( $selected_posts[0] );
		$post_type_rest_base = get_post_type_object( $first_post->post_type )->rest_base;

		$params = [
			'post_ids'            => $selected_posts,
			'target_languages'    => $target_languages,
			'llm_type'            => $llm_type,
			'llm_api_token'       => $llm_api_token,
			'post_type_rest_base' => $post_type_rest_base,
			'origin_language'     => Polylang_Helpers::get_default_language(),
			'wordpress_base_url'  => home_url(),
			'wordpress_app_pass'  => get_option( Settings_Page::OPTION_WP_APP_PASSWORD, '' ),
			'extra_prompt'        => get_option( Settings_Page::OPTION_EXTRA_PROMPT, '' ),
			'chunk_char_limit'    => get_option( Settings_Page::OPTION_CHUNK_CHAR_LIMIT, 500 ),
			'run-test'            => 'yes' === sanitize_text_field( $_POST['run_test_webhook'] ?? '' ) ? true : false,
		];

		$workflow_log_id = time();
		Workflow_Log_Crud::add_workflow_log( $workflow_log_id, $params );

		$n8n_result = N8N_Connection::send_to_n8n( $workflow_log_id, $params );

		if ( is_wp_error( $n8n_result ) ) {
			Workflow_Log_Crud::update_workflow_log_status( $workflow_log_id, 'failed', $n8n_result->get_error_message() );
			$error_message  = __( 'Failed to send posts to n8n for translation. Check n8n webhook URL and API keys.', 'coco-polylang-autotranslate' );
			$error_details  = 'Error code: ' . $n8n_result->get_error_code() . '\n';
			$error_details .= 'Error message: ' . $n8n_result->get_error_message();
			self::add_admin_notice( 'error', $error_message, $error_details );
		} else {
			$message = sprintf(
				/* translators: %1$s: comma-separated list of post IDs, %2$s: comma-separated list of target languages */
				__( 'Successfully sent posts %1$s for translation to %2$s.', 'coco-polylang-autotranslate' ),
				implode( ', ', $selected_posts ),
				implode( ', ', $target_languages )
			);
			$message .= ' ' . __( 'Check the workflow logs to see the evolution of the task.', 'coco-polylang-autotranslate' );
			$message .= sprintf(
				' <a href="%s" class="button button-secondary">%s</a>',
				admin_url( 'admin.php?page=coco-polylang-autotranslate-workflow-logs' ),
				__( 'View Workflow Logs', 'coco-polylang-autotranslate' )
			);
			self::add_admin_notice( 'success', $message );
			self::redirect_back();
			exit;
		}

		self::redirect_back( $extra_args_if_redirect_back );
		exit;
	}

	/**
	 * Handle the after update workflow log action.
	 *
	 * @param int   $workflow_log_id The workflow log ID.
	 * @param array $workflow_log The workflow log data.
	 * @return bool
	 */
	public static function handle_after_update_workflow_log( int $workflow_log_id, array $workflow_log ): bool {
		if ( ! isset( $workflow_log['params']['n8n-results'] ) ) {
			return false;
		}


		$n8n_results = $workflow_log['params']['n8n-results'];

		if ( ! is_array( $n8n_results ) ) {
			return false;
		}

		$target_post_ids = [];
		foreach ( $n8n_results as $result ) {
			if ( isset( $result['target_post_id'] ) ) {
				$target_post_ids[] = (int) $result['target_post_id'];
			}
		}

		$unique_target_post_ids = array_unique( $target_post_ids );

		foreach ( $unique_target_post_ids as $post_id ) {
			Chunk_Manager::save_post_content_from_chunks( $post_id );
		}

		return Workflow_Log_Crud::update_workflow_log_status( $workflow_log_id, 'wp-completed', 'All chunks have been processed and posts have been updated.' );
	}

	/**
	 * Handle the post type change action
	 *
	 * @return void
	 */
	public static function handle_change_post_type(): void {
		// Verify nonce and user capabilities
		$nonce = sanitize_text_field( $_POST['coco_pa_nonce'] ?? '' );
		if ( ! wp_verify_nonce( $nonce, 'coco_pa_change_post_type' )
				|| ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Access denied.', 'coco-polylang-autotranslate' ) );
		}

		// Get and validate the selected post type
		$selected_post_type = sanitize_text_field( $_POST['selected_post_type'] ?? '' );

		if ( empty( $selected_post_type ) ) {
			self::add_admin_notice( 'error', __( 'No post type selected.', 'coco-polylang-autotranslate' ) );
			self::redirect_back();
			return;
		}

		// Validate post type is in allowed list
		$valid_post_types = Actions_Page::get_valid_post_types();
		if ( ! in_array( $selected_post_type, $valid_post_types, true ) ) {
			self::add_admin_notice( 'error', __( 'Invalid post type selected.', 'coco-polylang-autotranslate' ) );
			self::redirect_back();
			return;
		}

		// Save the selected post type
		$updated = update_option( self::SELECTED_POST_TYPE_OPTION, $selected_post_type );

		if ( $updated ) {
			self::add_admin_notice(
				'success',
				sprintf(
					/* translators: %s: post type name */
					__( 'Post type successfully changed to "%s".', 'coco-polylang-autotranslate' ),
					$selected_post_type
				)
			);
		} else {
			// Option might not have changed or there was an error
			$current_value = get_option( self::SELECTED_POST_TYPE_OPTION, 'post' );
			if ( $current_value === $selected_post_type ) {
				self::add_admin_notice( 'info', __( 'Post type was already set to the selected value.', 'coco-polylang-autotranslate' ) );
			} else {
				self::add_admin_notice( 'error', __( 'Failed to update post type. Please try again.', 'coco-polylang-autotranslate' ) );
			}
		}

		self::redirect_back();
	}

	/**
	 * Get the currently selected post type
	 *
	 * @return string
	 */
	public static function get_selected_post_type(): string {
		return get_option( self::SELECTED_POST_TYPE_OPTION, 'post' );
	}

	/**
	 * Add an admin notice
	 *
	 * @param string $type Notice type (success, error, warning, info).
	 * @param string $message Notice message.
	 * @param string $details Optional details to display for admins.
	 * @return void
	 */
	private static function add_admin_notice( string $type = 'info', string $message = '', string $details = '' ): void {
		if ( 'error' === $type && ! empty( $details ) && current_user_can( 'manage_options' ) ) {
			$message .= '<br><br><strong>' . __( 'Admin Details:', 'coco-polylang-autotranslate' ) . '</strong><pre>' . esc_html( $details ) . '</pre>';
		}

		add_settings_error(
			'coco_pa_actions_notices',
			'post_type_change',
			$message,
			$type
		);
		set_transient( 'settings_errors', get_settings_errors(), 30 );
	}

	/**
	 * Redirect back to the actions page
	 *
	 * @param array $extra_query_args Optional array of additional query parameters.
	 * @return void
	 */
	private static function redirect_back( array $extra_query_args = [] ): void {
		$url = admin_url( 'admin.php?page=coco-polylang-autotranslate-actions&settings-updated=true' );
		if ( ! empty( $extra_query_args ) ) {
			$url = add_query_arg( $extra_query_args, $url );
		}
		wp_safe_redirect( $url );
		exit;
	}
}
